/* fif.c "Fully Illuminated Field" of Newtonian Telescopes
 *
 * My first version of program to determine fully illuminated field
 * in arcminutes (degrees) rather than the typical inches.
 * This one is a very simplified version, with no reference to
 * typical math functions because I had serious problems getting
 * the gcc compiler to do the right thing with math.h. Presuming
 * the same problems might appear on other machines, I put off
 * more elegant formulations. Portability!
 * There are no local includes to make this as simple and
 * universal as possible. I know this ain't good style, but it
 * is good KISS.
 * the program assumes you know your primary mirror size, focal
 * ratio, secondary minor axis, and intercept distance.
 * Units are irrelevant to this version, save that you must be
 * consistent (either use inches, millimeters, cubits or whatever
 * throughout).
 * The primary mirror size is its width.
 * The focal ratio is the focal length divided by primary width.
 * The secondary minor axis is the smallest width across the face.
 * The intercept is the distance from the center of the
 * secondary to the field stop position (roughly) of the eyepiece
 * when in the focuser (or, the distance from the secondary to
 * the focal plane of the primary when the telescope is assembled).
 * First it will ask you the primary size. Put the answer on the
 * same line as the question; just type it in and hit return. Etc.
 * Instructions will be printed at the beginning.
 * If you want to solve for field of view, you'll have to try
 * different secondary sizes in this version. Other revs will
 * be a bit more elaborate.
 * The idea is simple: the field of view is determined by the
 * width radius of secondary extending beyond the light cone at
 * intercept. This determines a "side" of a right triangle we're
 * interested in; the other is the focal length - intercept (or
 * distance from the primary to the secondary). Using arctan of
 * those two values, we get the max shift angle of the light cone
 * before it runs out of secondary. And in this dumbdown version,
 * it's interesting to note that at these angles the orignal ratio
 * (radius beyond cone/distance from mirror) and its arctan are
 * identical within our margin of error. So we skip the arctan
 * phase and simply use the ratio.
 * The margin of error is quite large for a number of reasons,
 * involving precision of construction, secondary adjustment
 * methods, offset errors (inevitable), limited secondary
 * size choices and etc mean the physical system being described
 * will not be any more accurate than these formulae...
 * Also for brevity, there is NO error checking (bad)
 * except for the commonly goofed-up focal ratio. 
 */
 
 
#include <stdio.h>
 
 
main()
{
/* there are more variables than strictly necessary (several
 * could serve double duty) but we're going to use them all
 * for clarity's sake 
 */
 
float primary,   /* mirror width */
fratio,          /* focal ratio (length/width) */
intercept,       /* distance from primary to focus */
secondary,       /* width of secondary at narrowest */
flength,         /* focal length (fratio * primary) */
longleg,         /* long leg of triangle (we want angle) */
shortleg,        /* short leg of triangle */
conic,           /* radius of light cone at intercept */
fudgeAtan,       /* short leg/long leg */
arcminutes,      /* just another way of looking at the field */
fif;             /* the traditional "fully illuminated field */
 
const float pi = 3.14159;   /* surely... */
float degree = 180 / pi;    /* converts radians to degrees */
 
char check;      /* readin for loop test */
int affirm;      /* test value for do loop */
 
/* first a printf to explain what to do */
 
printf("Fully Illuminated Field (executable binary)\n\n");
printf("You need to know four numbers to use this program:\n");
printf(" 1. Your primary size (width).\n");
printf(" 2. Your focal ratio (focal length/primary size).\n");
printf(" 3. Your diagonal minor axis (least width).\n");
printf(" 4. Your intercept distance (secondary to focal plane).\n");
printf(" The last is typically measured from the center of the\n");
printf(" secondary to the place in the focuser where the average\n");
printf(" eyepiece's field stop sits when focused. To be cautious,\n");
printf(" you may wish to use the eyepiece that comes to focus\n");
printf(" furthest out from the diagonal).\n");
printf("You can use any units you wish; the program works with inches,\n");
printf("millimeters, cubits, whatever. Just be consistent!\n\n");
printf("After answering, the program will ask if you want to kill it.\n");
printf("Type 'y' for yes or 'n' for no, but you must hit return.\n");
 
/* The entire program resides in a do loop designed to let the user
 * continue to use it, or kill it after any cycle 
 */
 
do
{
  printf("\nWhat is your primary size? ");
  scanf(" %f", &primary);
 
/* as you see above, we read in the primary size and took the
 * user's word for it. But not the next time; we're going to
 * check to see if the value is reasonable, and ask if he really
 * meant it. The do loop will execute once, then die if the
 * conditions are met. The condition is for the variable
 * "affirm" to be nonzero -- and it is initially set to 1. So
 * they have to be a little outrageous to set it to zero, but
 * then again, we don't know everything, so we give them a
 * chance to back out 
 */
 
do
{
  affirm = 1;
  printf("What is your focal ratio? ");
  scanf(" %f", &fratio );
  if( fratio < 2 )
  {
    printf("That's an awfully small focal ratio.\n");
    printf("Are you sure? -- y for yes, n for no. ");
    getchar();
    check = getchar();
    if( check != 'y' && check != 'Y')
      {
        affirm = 0;
        printf("I'll ask again: \n");
      }
  }
  else if( fratio > 30 )
  {
    printf("That's an awfully big focal ratio.\n");
    printf("Are you sure? -- y for yes, n for no. ");
    getchar();
    check = getchar();
     if( check != 'y' && check != 'Y')
       {
         affirm = 0;
         printf("I'll ask again, affirm = %d: \n", affirm );
       }
  }
} while ( affirm == 0 );
 
/* it's clear we could write a struct or class to manage the above
 * function and carry it forward; I might just do that in c, though
 * I want to reserve the fun stuff for the c++ version. Still, a
 * little bit... 
 */
 
printf("What is your minor axis? ");
scanf(" %f", &secondary);
 
printf("What is your intercept? ");
scanf(" %f", &intercept );
 
/* okay, now for the "math." first, the focal length */
 
flength = primary * fratio;
printf("\nYour focal length is: %.1f", flength );
 
/* Next we'll figure out the long leg of the "field of view"
 * triangle, which is also the distance from the primary to
 * the center of the secondary 
 */
 
longleg = flength - intercept;
printf("\nYour distance from primary to secondary is: %.1f", longleg );
 
/* Now to figure the circle defined by intersecting the cone
 * at the middle of the secondary mirror formed by the light
 * coming from the primary mirror to focus. This is a simple
 * ratio of the distance from the center of the secondary to
 * the focal point (intercept) divided by the focal length
 * (which is the length of the entire "focal cone." Since the
 * base of the triangle is the diameter of the mirror, we
 * multiply it by that ratio 
 */
 
conic = ( intercept / flength ) * primary;
printf("\nYour on-axis illuminated spot is: %.3f", conic );
 
/* Now we need to know how much the secondary overhangs (is
 * larger than) the conic spot at the center of the secondary.
 * That's simple; it's the secondary minor axis (the diameter
 * of the "circle" the secondary presents to the cone) less
 * the diameter of the cone at that point. This, of course, is
 * diameter - diameter, but we only want one side (which is
 * radius - radius) so we divide it by two */
 
shortleg = ( secondary - conic ) / 2;
printf("\nYour secondary ''overhang'' is %.3f", shortleg );
 
/* now we have my biggest crime: since the arctangent (in radians)
 * of the ratio of the shortleg divided by the longleg is almost
 * exactly the same as the ratio itself, I just use the ratio. This
 * is true to several decimal places, so for our purposes it's okay,
 * and I'm able to avoid including math.h, which was freaking out
 * the compiler 
 */
 
fudgeAtan = shortleg / longleg;
 
/* the next part is really easy: we just got our answer in radians,
 * and we want degrees (since our 'customer' is going to want to
 * convert to degrees in order to see how his eyepieces will work 
 */
 
fif = fudgeAtan * degree;
printf("\n\nThe Fully Illuminated Field of view is %.3f degrees.\n", fif );
 
/* and as a glissando, we toss in the result in arcminutes. For those
 * who don't know, there are 60 arcminutes in every degree...
 */
 
arcminutes = 60 * fif;
 
printf("The Fully Illuminated Field in arcminutes is: ");
printf("%.1f\n\n", arcminutes );
 
printf("Well, there you have it.\n");
printf("Want to kill me? -- y for yes, n for no. ");
getchar();
check = getchar();
if( check != 'y' && check != 'Y')
  {
    affirm = 0;
    printf("\nThen we'll start all over again: \n");
  }
} while ( affirm == 0 );
 
return 0;
}

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * Or, as of the time of this writing, consult: 
 * http://www.gnu.org/copyleft/gpl.html

 * First published 1999
 */